#!/bin/bash
################# DESCRIPTION ###################
#                                               #
#################################################
#············· EDITABLE·VARIABLES···············#
USERNAME="YourUserName"
PASSWORD="YourPassword"
COMMUNITY="YourCommunity"
ROOM_ID="1700000000013628"
LOCAL_PATH="/home/caveau/uploads"
BASE_URL="https://app.digitalcaveau.com"
#··········· END·EDITABLE·VARIABLES· ···········#

# Help
usage() {
  echo "$(basename "$0") is a simple script that uploads files into Caveau platform."
  echo "Configuration can be done inside the script by changing the values of the editable variables, "
  echo "Otherwise by giving in input positional parameters as showed in the Usage section."
  echo ""
  echo "Usage : $(basename "$0") [-v] <USERNAME> <PASSWORD> <COMMUNITY> <ROOM_ID> <LOCAL_PATH>"
  echo ""
  echo -e "Positional parameters"
  echo -e " <USERNAME>\\t the username used for log in to Caveau"
  echo -e " <PASSWORD>\\t the password of the above specified user"
  echo -e " <COMMUNITY>\\t the community login name"
  echo -e " <ROOM_ID>\\t the room id inside Caveau community where files will be uploaded,"
  echo -e " \\t \\t and it can be read from the address bar (URL) of your browser when room is open"
  echo -e " <LOCAL_PATH>\\t the folder in the local file system that will be processed"
  echo -e " \\t \\t Please note: two subfolders will be created at this location, and all processed "
  echo -e " \\t \\t files will be moved into one of those depending if upload succeeded or failed"
  echo ""
  echo -e "Options"
  echo -e " -h \\tshow this help text"
  echo -e " -v \\tverbose output"
}

while getopts ':hv' option; do
  case "$option" in
    h) usage; 	exit;;
    v) printf "activating verbosity"; VERBOSE=1;;
    \?) printf "illegal option: -%s\\n" "$OPTARG" >&2; usage >&2; exit 1;;
  esac
done
shift $((OPTIND - 1))

# Lock script if it's already running
lockdir=/tmp/$(basename "$0").lock
if mkdir "$lockdir"
then
  # directory did not exist, but was created successfully
  echo >&2 "successfully acquired lock: $lockdir"
  # remove directory when script finishes
  trap 'rm -rf "$lockdir"' 0
  # continue script
else
  echo >&2 "cannot acquire lock, script is already running, giving up on $lockdir"
  exit 0
fi

# Check positional parameters, substitute variables if correct number of args are given.
if [[ "$#" -ne 5 ]] && [[ "$#" -ne 0 ]]; then
  echo "Illegal number of parameters"
  exit 1
elif [[ $# -eq 0 ]]; then
  echo "No arguments provided, using the default configuration values set inside the script."
elif [[ $# -eq 5 ]]; then
  if [ -z ${1+x} ]; then echo "positional parameter '$1' is unset, using the default"; else echo "var is set to '$1'" && USERNAME=$1; fi
  if [ -z ${2+x} ]; then echo "positional parameter '$2' is unset, using the default"; else echo "var is set to '$2'" && PASSWORD=$2; fi
  if [ -z ${3+x} ]; then echo "positional parameter '$3' is unset, using the default"; else echo "var is set to '$3'" && COMMUNITY=$3; fi
  if [ -z ${4+x} ]; then echo "positional parameter '$4' is unset, using the default"; else echo "var is set to '$4'" && ROOM_ID=$4; fi
  if [ -z ${5+x} ]; then echo "positional parameter '$5' is unset, using the default"; else echo "var is set to '$5'" && LOCAL_PATH=$5; fi
fi

#··· Start process
if ((VERBOSE)); then
  set -o xtrace
fi

# Perform authentication request
AUTH="$(curl -s -k --data "client_id=bash&client_secret=secret&grant_type=password&username=$USERNAME&password=$PASSWORD&community=$COMMUNITY" \
  $BASE_URL/agora-api/api/v1/auth \
  --header "Content-Type: application/x-www-form-urlencoded")"
#echo $AUTH

# Check authentication
if [[ $AUTH = *"error_description"* ]]; then
  echo "$AUTH" | sed -E 's/.*error_description\"\:\"([^\"]+)\".*/\1/'
  exit 1
elif [[ $AUTH = *"access_token"* ]]; then
  TOKEN=`echo "$AUTH" | sed -E 's/.*access_token\"\:\"([^\"]+)\".*/\1/'`
  #echo Your token: $TOKEN
fi

# Check directories exists
if [[ -d "$LOCAL_PATH" ]]; then
  # if subfolders not exists create them
  mkdir -p "$LOCAL_PATH"/backup
  mkdir -p "$LOCAL_PATH"/errors
else
  echo Local path not found
  exit 1
fi

find "$LOCAL_PATH" -maxdepth 1 -type f | while read -r FILE; 
do
  if [[ -f $FILE ]]; then
    FILENAME=$(basename "$FILE")
    CONTENT_TYPE=$(file -b --mime-type "$FILE")
    UPLOAD="$(curl -w %{response_code} -s --output /dev/null -k -X POST -H "Authorization: Bearer $TOKEN" \
      -F "Resource={\"inheritSettings\":true}" \
      -F "file=@$FILE;filename=$FILENAME;type=$CONTENT_TYPE;" \
      "$BASE_URL/agora-api/api/v1/rooms/$ROOM_ID/files")"
    
    # Check upload status
    if [[ ${UPLOAD} = *"200"* ]] || [[ ${UPLOAD} = *"201"* ]]; then
      echo "$FILENAME: [$UPLOAD] successfully transferred to Caveau..."
      mv "$FILE" "$LOCAL_PATH"/backup/
    else
      echo "$FILENAME: could not be transferred to Caveau (Error $UPLOAD)"
      mv "$FILE" "$LOCAL_PATH"/errors/
    fi
  fi
done

# Logout
LOGOUT="$(curl -s -k $BASE_URL/agora-api/api/v1/logout -H "Authorization: Bearer $TOKEN")"
if ((VERBOSE)); then
  echo "Logout status $LOGOUT"
fi
